<?php

class Simproba_Pricing {
    private $coupon_code = 'simproba_role_discount';

    public function __construct() {
        simproba_debug_log('Simproba_Pricing class initialized');
        add_filter('woocommerce_product_get_price', array($this, 'modify_product_price'), 100, 2);
        add_filter('woocommerce_product_get_sale_price', array($this, 'modify_product_price'), 100, 2);
        add_filter('woocommerce_product_variation_get_price', array($this, 'modify_product_price'), 100, 2);
        add_filter('woocommerce_product_variation_get_sale_price', array($this, 'modify_product_price'), 100, 2);
        add_action('woocommerce_cart_loaded_from_session', array($this, 'apply_role_coupon'), 100);
        add_action('woocommerce_cart_contents', array($this, 'force_cart_recalculation'), 100);
        add_action('woocommerce_checkout_update_order_meta', array($this, 'save_coupon_to_order'), 100, 2);
        add_action('init', array($this, 'create_role_coupon'), 10);
    }

    public function create_role_coupon() {
        simproba_debug_log('R1: Début de create_role_coupon');
        if (!is_user_logged_in()) {
            simproba_debug_log('R2: Utilisateur non connecté, coupon non créé');
            return;
        }
        $user = wp_get_current_user();
        $user_roles = $user->roles;
        simproba_debug_log('R3: Rôles utilisateur: %s', print_r($user_roles, true));
        global $wpdb;
        $table_name = $wpdb->prefix . 'simproba_settings';
        $rules = $wpdb->get_results($wpdb->prepare("SELECT role, enable, discount_type, discount_value, categories FROM $table_name WHERE post_type = %s AND enable = %s", 'product', 'yes'));
        simproba_debug_log('R4: Règles récupérées: %s', print_r($rules, true));
        if (empty($rules)) {
            simproba_debug_log('R5: Aucune règle active, coupon non créé');
            return;
        }
        $coupon_exists = get_posts([
            'post_type' => 'shop_coupon',
            'post_status' => 'publish',
            'name' => $this->coupon_code,
            'numberposts' => 1,
        ]);
        if (!empty($coupon_exists)) {
            simproba_debug_log('R6: Coupon %s déjà existant', $this->coupon_code);
            return;
        }
        foreach ($user_roles as $role) {
            foreach ($rules as $rule) {
                if ($rule->role !== $role) {
                    simproba_debug_log('R7: Rôle %s ne correspond pas à la règle pour %s', $role, $rule->role);
                    continue;
                }
                $discount_value = floatval($rule->discount_value);
                $rule_categories = $rule->categories ? json_decode($rule->categories, true) : [];
                simproba_debug_log('R8: Catégories de la règle pour rôle %s: %s', $role, print_r($rule_categories, true));
                $coupon = array(
                    'post_title' => $this->coupon_code,
                    'post_content' => '',
                    'post_status' => 'publish',
                    'post_author' => 1,
                    'post_type' => 'shop_coupon',
                );
                $new_coupon_id = wp_insert_post($coupon);
                if ($new_coupon_id) {
                    update_post_meta($new_coupon_id, 'discount_type', $rule->discount_type === 'percentage' ? 'percent' : ($rule->discount_type === 'fixed' ? 'fixed_cart' : 'fixed_product'));
                    update_post_meta($new_coupon_id, 'coupon_amount', $discount_value);
                    update_post_meta($new_coupon_id, 'individual_use', 'no');
                    update_post_meta($new_coupon_id, 'product_ids', '');
                    update_post_meta($new_coupon_id, 'exclude_product_ids', '');
                    update_post_meta($new_coupon_id, 'usage_limit', '0');
                    update_post_meta($new_coupon_id, 'usage_limit_per_user', '0');
                    update_post_meta($new_coupon_id, 'limit_usage_to_x_items', '0');
                    update_post_meta($new_coupon_id, 'free_shipping', 'no');
                    update_post_meta($new_coupon_id, 'date_expires', '');
                    update_post_meta($new_coupon_id, 'minimum_amount', '0');
                    update_post_meta($new_coupon_id, 'maximum_amount', '');
                    update_post_meta($new_coupon_id, 'customer_email', '');
                    update_post_meta($new_coupon_id, 'exclude_sale_items', 'no');
                    if (!empty($rule_categories) && is_array($rule_categories)) {
                        update_post_meta($new_coupon_id, 'product_categories', $rule_categories);
                    }
                    simproba_debug_log('R9: Coupon %s créé avec succès, ID: %d, type: %s, montant: %f', $this->coupon_code, $new_coupon_id, $rule->discount_type, $discount_value);
                    return;
                } else {
                    simproba_debug_log('R10: Échec de la création du coupon %s', $this->coupon_code);
                }
            }
        }
        simproba_debug_log('R11: Aucune règle de coupon applicable');
    }

    public function apply_role_coupon($cart) {
        simproba_debug_log('C1: Début de apply_role_coupon, is_cart: %s, is_checkout: %s, DOING_AJAX: %s', is_cart() ? 'yes' : 'no', is_checkout() ? 'yes' : 'no', defined('DOING_AJAX') ? 'yes' : 'no');
        if (!is_user_logged_in()) {
            simproba_debug_log('C2: Utilisateur non connecté, coupon non appliqué');
            return;
        }
        $user = wp_get_current_user();
        $user_roles = $user->roles;
        simproba_debug_log('C3: Rôles utilisateur: %s', print_r($user_roles, true));
        global $wpdb;
        $table_name = $wpdb->prefix . 'simproba_settings';
        $rules = $wpdb->get_results($wpdb->prepare("SELECT role, enable, discount_type, discount_value FROM $table_name WHERE post_type = %s AND enable = %s", 'product', 'yes'));
        simproba_debug_log('C4: Règles récupérées: %s', print_r($rules, true));
        if (empty($rules)) {
            simproba_debug_log('C5: Aucune règle active, coupon non appliqué');
            return;
        }
        foreach ($user_roles as $role) {
            foreach ($rules as $rule) {
                if ($rule->role !== $role) {
                    simproba_debug_log('C6: Rôle %s ne correspond pas à la règle pour %s', $role, $rule->role);
                    continue;
                }
                if (!is_a($cart, 'WC_Cart')) {
                    simproba_debug_log('C7: Erreur - $cart n\'est pas un objet WC_Cart, type: %s', gettype($cart));
                    return;
                }
                if (!$cart->has_discount($this->coupon_code)) {
                    $result = $cart->apply_coupon($this->coupon_code);
                    simproba_debug_log('C8: Coupon %s appliqué au panier, résultat: %s', $this->coupon_code, $result ? 'succès' : 'échec');
                } else {
                    simproba_debug_log('C9: Coupon %s déjà appliqué', $this->coupon_code);
                }
                simproba_debug_log('C10: Total du panier après coupon: %f', $cart->get_total('edit'));
                return;
            }
        }
        simproba_debug_log('C11: Aucune règle de coupon applicable');
    }

    public function force_cart_recalculation($cart) {
        simproba_debug_log('F1: Début de force_cart_recalculation, is_cart: %s, is_checkout: %s, DOING_AJAX: %s', is_cart() ? 'yes' : 'no', is_checkout() ? 'yes' : 'no', defined('DOING_AJAX') ? 'yes' : 'no');
        if (!is_user_logged_in()) {
            simproba_debug_log('F2: Utilisateur non connecté, aucun recalcul');
            return;
        }
        if (!is_a($cart, 'WC_Cart')) {
            simproba_debug_log('F3: Erreur - $cart n\'est pas un objet WC_Cart, type: %s', gettype($cart));
            return;
        }
        if ($cart->has_discount($this->coupon_code)) {
            $cart->calculate_totals();
            simproba_debug_log('F4: Panier recalculé, total: %f', $cart->get_total('edit'));
        } else {
            simproba_debug_log('F5: Coupon %s non appliqué, pas de recalcul', $this->coupon_code);
        }
    }

    public function save_coupon_to_order($order_id, $data) {
        simproba_debug_log('O1: save_coupon_to_order pour commande ID: %d', $order_id);
        $order = wc_get_order($order_id);
        if (in_array($this->coupon_code, $order->get_coupon_codes(), true)) {
            $order->add_meta_data('simproba_role_discount', $this->coupon_code, true);
            $order->save();
            simproba_debug_log('O2: Coupon %s enregistré dans la commande ID %d', $this->coupon_code, $order_id);
        } else {
            simproba_debug_log('O3: Aucun coupon %s appliqué à la commande ID %d', $this->coupon_code, $order_id);
        }
    }

    public function modify_product_price($price, $product) {
        simproba_debug_log('P1: modify_product_price pour produit ID: %d, Titre: %s, Prix initial: %f', $product->get_id(), $product->get_title(), $price);
        $base_price = floatval($product->get_regular_price());
        if (!$base_price) {
            simproba_debug_log('P2: Prix de base non défini pour produit ID: %d, prix inchangé: %f', $product->get_id(), $price);
            return $price;
        }
        if (!is_user_logged_in()) {
            simproba_debug_log('P3: Utilisateur non connecté, prix inchangé: %f', $price);
            return $price;
        }
        $user = wp_get_current_user();
        $user_roles = $user->roles;
        simproba_debug_log('P4: Rôles utilisateur: %s', print_r($user_roles, true));
        global $wpdb;
        $table_name = $wpdb->prefix . 'simproba_settings';
        $rules = $wpdb->get_results($wpdb->prepare("SELECT role, enable, discount_type, discount_value, categories FROM $table_name WHERE post_type = %s AND enable = %s", 'product', 'yes'));
        simproba_debug_log('P5: Règles récupérées: %s', print_r($rules, true));
        if (empty($rules)) {
            simproba_debug_log('P6: Aucune règle active, prix inchangé: %f', $price);
            return $price;
        }
        $product_categories = wp_get_post_terms($product->get_id(), 'product_cat', ['fields' => 'ids']);
        if (is_wp_error($product_categories)) {
            simproba_debug_log('P7: Erreur dans wp_get_post_terms: %s', $product_categories->get_error_message());
            return $price;
        }
        $product_categories = array_map('intval', $product_categories);
        simproba_debug_log('P8: Catégories du produit ID %d: %s', $product->get_id(), print_r($product_categories, true));
        foreach ($user_roles as $role) {
            foreach ($rules as $rule) {
                if ($rule->role !== $role) {
                    simproba_debug_log('P9: Rôle %s ne correspond pas à la règle pour %s', $role, $rule->role);
                    continue;
                }
                $rule_categories = $rule->categories ? json_decode($rule->categories, true) : [];
                simproba_debug_log('P10: Catégories de la règle pour rôle %s: %s', $role, print_r($rule_categories, true));
                if (!is_array($rule_categories)) {
                    simproba_debug_log('P11: Erreur - rule_categories n\'est pas un tableau, règle ignorée pour le rôle %s', $role);
                    continue;
                }
                $rule_categories = array_map('intval', $rule_categories);
                if (!empty($rule_categories)) {
                    $has_matching_category = false;
                    foreach ($product_categories as $cat_id) {
                        if (in_array($cat_id, $rule_categories, true)) {
                            $has_matching_category = true;
                            simproba_debug_log('P12: Catégorie correspondante trouvée: %d', $cat_id);
                            break;
                        }
                    }
                    if (!$has_matching_category) {
                        simproba_debug_log('P13: Aucune catégorie correspondante, règle ignorée pour le rôle %s', $role);
                        continue;
                    }
                } else {
                    simproba_debug_log('P14: Aucune catégorie spécifiée, remise appliquée à tous les produits pour le rôle %s', $role);
                }
                $discount_value = floatval($rule->discount_value);
                simproba_debug_log('P15: Application de la remise pour le rôle %s: type=%s, valeur=%f', $role, $rule->discount_type, $discount_value);
                switch ($rule->discount_type) {
                    case 'percentage':
                        $new_price = $base_price * (1 - $discount_value / 100);
                        simproba_debug_log('P16: Remise en pourcentage appliquée, nouveau prix=%f', $new_price);
                        return $new_price;
                    case 'fixed':
                        $new_price = max(0, $base_price - $discount_value);
                        simproba_debug_log('P17: Remise fixe appliquée, nouveau prix=%f', $new_price);
                        return $new_price;
                    case 'fixed_price':
                        $new_price = $discount_value;
                        simproba_debug_log('P18: Prix fixe appliqué, nouveau prix=%f', $new_price);
                        return $new_price;
                    default:
                        simproba_debug_log('P19: Type de remise inconnu: %s', $rule->discount_type);
                        break;
                }
            }
        }
        simproba_debug_log('P20: Aucune règle applicable, prix inchangé: %f', $price);
        return $price;
    }
}

new Simproba_Pricing();